#!/usr/bin/env python
# coding: utf-8
"""Oktizer: Change pads and vias to octagons."""
# 🄍 CC0 public domain

# Update PCB first:
#     Tools → Cleanup Tracks & Vias…   → ☒ Merge co-linear tracks
#     Edit  → Cleanup Tracks and Vias… → ☒ Merge overlapping segments

from __future__ import division, unicode_literals, print_function
from math import ceil, cos, sin
from collections import defaultdict as ddict

# FromMM()/FromMils() don't round symmetrically; mmToIU()/MilsToIU() take single argument only
def mm2iu(*mm):   return _u2iu(IU_PER_MM, mm)
def mil2iu(*mil): return _u2iu(IU_PER_MILS, mil)
def _u2iu(c, u):
    iu = (int(round(c * i)) for i in u)
    return next(iu) if len(u) == 1 else tuple(iu)

try:
    from functools import cache
except ImportError:
    def cache(function):
        from functools import wraps
        from collections import defaultdict
        class cachedict(defaultdict):
            def __missing__(self, key):
                self[key] = val = self.default_factory(*key)
                return val
        cache = cachedict(function)
        @wraps(function)
        def wrapper(*args):
            return cache[args]
        wrapper.cache_clear = cache.clear
        return wrapper

try:
    from pcbnew import (  # 7.0
        ActionPlugin, GetBoard, pcbIUScale,
        PAD, PCB_TRACK, PCB_VIA, ZONE,  # ZONE_FILLER, TEARDROP_TYPE,
        VECTOR2I as wxPoint, VECTOR_VECTOR2I as wxPoint_Vector, VECTOR2I as wxSize,
        CUST_PAD_SHAPE_IN_ZONE_CONVEXHULL,
        ZONE_SETTINGS,  # .SMOOTHING_NONE
        PAD_SHAPE_CIRCLE, PAD_SHAPE_RECT, PAD_SHAPE_OVAL,  # PAD_SHAPE_TRAPEZOID,
        PAD_SHAPE_ROUNDRECT, PAD_SHAPE_CHAMFERED_RECT, PAD_SHAPE_CUSTOM,
        PAD_ATTRIB_PTH, PAD_ATTRIB_SMD, PAD_ATTRIB_CONN, PAD_ATTRIB_NPTH,
        VIATYPE_THROUGH, ZONE_CONNECTION_FULL,
        ZONE_BORDER_DISPLAY_STYLE_NO_HATCH, ISLAND_REMOVAL_MODE_NEVER,
        ZONE_FILL_MODE_POLYGONS, ZONE_THICKNESS_MIN_VALUE_MM,
        BOARD, FOOTPRINT, FP_EXCLUDE_FROM_POS_FILES, FP_EXCLUDE_FROM_BOM, FP_SHAPE,
        PCB_SHAPE, SHAPE_T_CIRCLE, SHAPE_T_POLY, SHAPE_T_RECT,
        IO_MGR, LIB_ID, LSET, F_Cu, B_Cu, F_SilkS, F_Fab,
    )
    IU_PER_MM, IU_PER_MILS = pcbIUScale.IU_PER_MM, pcbIUScale.IU_PER_MILS
    FP_VIRTUAL = FP_EXCLUDE_FROM_POS_FILES | FP_EXCLUDE_FROM_BOM
    RECT_CHAMFER_ALL = 0xf  # RECT_CHAMFER_POSITIONS.ALL
    ZONE_THICKNESS_MIN_VALUE = mm2iu(ZONE_THICKNESS_MIN_VALUE_MM)
except ImportError:
    try:
        from pcbnew import (  # 6.0
            ActionPlugin, GetBoard, IU_PER_MM, IU_PER_MILS,
            PAD, PCB_TRACK, PCB_VIA, ZONE, ZONE_FILLER,
            wxPoint, wxPoint_Vector, wxSize,
            CUST_PAD_SHAPE_IN_ZONE_CONVEXHULL,
            ZONE_SETTINGS,  # .SMOOTHING_NONE
            PAD_SHAPE_CIRCLE, PAD_SHAPE_RECT, PAD_SHAPE_OVAL,  # PAD_SHAPE_TRAPEZOID,
            PAD_SHAPE_ROUNDRECT, PAD_SHAPE_CHAMFERED_RECT, PAD_SHAPE_CUSTOM,
            PAD_ATTRIB_PTH, PAD_ATTRIB_SMD, PAD_ATTRIB_CONN, PAD_ATTRIB_NPTH,
            VIATYPE_THROUGH, ZONE_CONNECTION_FULL,
            ZONE_BORDER_DISPLAY_STYLE_NO_HATCH, ISLAND_REMOVAL_MODE_NEVER,
            ZONE_FILL_MODE_POLYGONS, ZONE_THICKNESS_MIN_VALUE_MIL,
            BOARD, FOOTPRINT, FP_EXCLUDE_FROM_POS_FILES, FP_EXCLUDE_FROM_BOM, FP_SHAPE,
            PCB_SHAPE, SHAPE_T_CIRCLE, SHAPE_T_POLY, SHAPE_T_RECT,
            IO_MGR, LIB_ID, LSET, F_Cu, B_Cu, F_SilkS, F_Fab,
        )
        FP_VIRTUAL = FP_EXCLUDE_FROM_POS_FILES | FP_EXCLUDE_FROM_BOM
        RECT_CHAMFER_ALL = 0xf
        ZONE_THICKNESS_MIN_VALUE = mil2iu(ZONE_THICKNESS_MIN_VALUE_MIL)
    except ImportError:
        from pcbnew import (  # 5.1
            ActionPlugin, GetBoard, IU_PER_MM, IU_PER_MILS, 
            D_PAD as PAD, TRACK as PCB_TRACK, VIA as PCB_VIA, ZONE_CONTAINER as ZONE, ZONE_FILLER,
            wxPoint, wxPoint_Vector, wxSize,
            CUST_PAD_SHAPE_IN_ZONE_CONVEXHULL,
            ZONE_SETTINGS,  # .SMOOTHING_NONE
            PAD_SHAPE_CIRCLE, PAD_SHAPE_RECT, PAD_SHAPE_OVAL,  # PAD_SHAPE_TRAPEZOID,
            PAD_SHAPE_ROUNDRECT, PAD_SHAPE_CUSTOM,
            PAD_ATTRIB_STANDARD as PAD_ATTRIB_PTH, PAD_ATTRIB_CONN, PAD_ATTRIB_SMD,
            PAD_ATTRIB_HOLE_NOT_PLATED as PAD_ATTRIB_NPTH,
            VIA_THROUGH as VIATYPE_THROUGH, PAD_ZONE_CONN_FULL as ZONE_CONNECTION_FULL,
            ZFM_POLYGONS as ZONE_FILL_MODE_POLYGONS, ZONE_THICKNESS_MIN_VALUE_MIL,
            BOARD, MODULE as FOOTPRINT, MOD_VIRTUAL as FP_VIRTUAL, EDGE_MODULE as FP_SHAPE,
            DRAWSEGMENT as PCB_SHAPE, S_CIRCLE as SHAPE_T_CIRCLE, S_POLYGON as SHAPE_T_POLY, S_RECT as SHAPE_T_RECT,
            IO_MGR, LIB_ID, LSET, F_Cu, B_Cu, F_SilkS, F_Fab,
        )
        ZONE_BORDER_DISPLAY_STYLE_NO_HATCH = ZONE.NO_HATCH
        ISLAND_REMOVAL_MODE_NEVER = PAD_SHAPE_CHAMFERED_RECT = RECT_CHAMFER_ALL = NotImplemented
        ZONE_THICKNESS_MIN_VALUE = mil2iu(ZONE_THICKNESS_MIN_VALUE_MIL)


__version__ = "0.7.0"

# twice the radius of rounded corners
#STROKE_WIDTH = 0              # rejected by Copper Zone Properties dialog
#STROKE_WIDTH = ZONE_THICKNESS_MIN_VALUE  # workaround
#STROKE_WIDTH = mm2iu(1)       # mil2iu(40)
STROKE_WIDTH = mm2iu(0.1)      # mil2iu(4)

# octagon definition in terms of a chamfered square/rectangle
#CHAMFER_RATIO =             (.5**.5 - .5,) * 2  # √½ - ½ ≈ 20.71% (side ratio 2÷1)
#CHAMFER_RATIO =          (0.25,)           * 2  #      ¼ = 25%   (isogonal octagon)
#CHAMFER_RATIO = (1 - .5**.5,)              * 2  # 1 - √½ ≈ 29.29% (regular octagon)
CHAMFER_RATIO = (1 - .5**.5, .5**.5 - .5)        # 1 - √½ ≈ 29.29% (regular octagon), √½ - ½ ≈ 20.71% (side ratio 2÷1)

# arbitrarily chosen initial markers
#ZONE_PRIORITY = 88  # KiCad 5.1 Copper Zone Properties dialog reduces priority to 100
ZONE_PRIORITY = 22222           # KiCad 7.0.0 uses 30000+, Teardrops uses 16962
ZONE_NAME     = "$oktizer$"     # KiCad 7.0.0 uses $teardrop_{padvia,track}$ informally only; see SetTeardropAreaType()


def chamferRatio(width=1, height=1):
    """Return appropriate chamfer ratio for square or rectangle."""
    return CHAMFER_RATIO[not 0.95 < height/width < 1.05]


@cache
def octastar(width, height, stroke=0):
    """Return points of concave octagonal star (open 16-gon)."""
    c = min(width, height) * chamferRatio(width, height)
    r, x, y, z = (i * 0.5 for i in (stroke, width, height, width+height))
    d = c + r * 2**.5  # √2
    e = d - r
    v = [j for i in map(ceil, (x-e, x-r, y-e, y-r, z-d, 0)) for j in (i, -i)]
    return tuple(wxPoint(v[i[0]], v[i[1]]) for i in octastar.idxtab)

octastar.idxtab = (
    (8, 10), (2, 4), (2, 6), (0, 6), (10, 8), (1, 6), (3, 6), (3, 4),
    (9, 10), (3, 5), (3, 7), (1, 7), (10, 9), (0, 7), (2, 7), (2, 5),
)


def oktize(board=None):
    """Change pads and vias to octagons."""
    if board is None:
        board = GetBoard()

    # setup zone and rule area (keepout zone) with defined defaults once
    z = ZONE(board)
    try:
        z.SetAssignedPriority(ZONE_PRIORITY)
        #z.SetTeardropAreaType(TEARDROP_TYPE.TD_VIAPAD)  # TEARDROP_TYPE.TD_UNSPECIFIED
    except AttributeError:
        z.SetPriority(ZONE_PRIORITY)
    z.SetIsFilled(False)
    z.SetFillMode(ZONE_FILL_MODE_POLYGONS)  # no hatch
    z.SetHatchStyle(ZONE_BORDER_DISPLAY_STYLE_NO_HATCH)
    z.SetMinThickness(STROKE_WIDTH)  # ignored for rule area/keepout
    z.SetPadConnection(ZONE_CONNECTION_FULL)  # no thermal
    z.SetCornerSmoothingType(ZONE_SETTINGS.SMOOTHING_NONE)
    z.SetCornerRadius(0)  # ignored for rule area/keepout
    z.SetDoNotAllowCopperPour(False)
    z.SetDoNotAllowVias(False)
    z.SetDoNotAllowTracks(False)
    try:
        z.SetHV45(True)  # GUI
    except AttributeError:
        pass
    try:
        z.SetLocalClearance(0)
        z.SetIsRuleArea(False)
        z.SetDoNotAllowPads(False)
        z.SetDoNotAllowFootprints(False)
        z.SetIslandRemovalMode(ISLAND_REMOVAL_MODE_NEVER)
        z.SetZoneName(ZONE_NAME)
    except AttributeError:
        z.SetZoneClearance(0)
        z.SetIsKeepout(False)
    zonedefaults = ZONE(z)
    try:
        z.SetIsRuleArea(True)
    except AttributeError:
        z.SetIsKeepout(True)
    z.SetDoNotAllowCopperPour(True)
    z.SetMinThickness(0)  # ignored for rule area/keepout
    keepoutdefaults = ZONE(z)


    # collect all board items to find out if any and which have been selected
    padAttributes = PAD_ATTRIB_PTH, PAD_ATTRIB_SMD, PAD_ATTRIB_CONN, PAD_ATTRIB_NPTH,
    padShapes = PAD_SHAPE_CIRCLE, PAD_SHAPE_OVAL, PAD_SHAPE_RECT, PAD_SHAPE_ROUNDRECT,  # PAD_SHAPE_CUSTOM,
    viaTypes = VIATYPE_THROUGH,

    connectivity = board.GetConnectivity()
    connections = ddict(lambda: ddict(list))  # connections[padOrVia][layer] = [*segments]
    zones = ddict(list)
    shapes = []
    notselected = True

    # collect selected or all pads
    for p in board.GetPads():
        if notselected and p.IsSelected():
            notselected = False
            connections.clear()
        if notselected or p.IsSelected():
            if p.GetAttribute() in padAttributes and p.GetShape() in padShapes:
                connections[p]  # autovivification

    # collect selected or all vias
    for t in board.GetTracks():
        if notselected and t.IsSelected():
            notselected = False
            connections.clear()
        if notselected or t.IsSelected():
            if type(t) is PCB_VIA and t.GetViaType() in viaTypes:
                connections[t]  # autovivification

    # collect selected or all zones
    try:
        for z in (board.GetArea(i) for i in range(board.GetAreaCount())):
            if notselected and z.IsSelected():
                notselected = False
                connections.clear()
            if not z.GetIsRuleArea():
                zones[z.GetNetCode()].append(z)
    except AttributeError:
        for z in (board.GetArea(i) for i in range(board.GetAreaCount())):
            if notselected and z.IsSelected():
                notselected = False
                connections.clear()
            if not z.GetIsKeepout():
                zones[z.GetNetCode()].append(z)

    # collect selected or all board circle/rect shapes
    for d in board.GetDrawings():
        if notselected and d.IsSelected():
            notselected = False
            connections.clear()
            del shapes[:]
        if notselected or d.IsSelected():
            if type(d) is PCB_SHAPE and d.GetShape() in (SHAPE_T_CIRCLE, SHAPE_T_RECT):
                shapes.append(d)


    # change circles/rects into octagons only if explicitly selected and no pads/vias/zones selected
    if not (notselected or connections):
        for s in shapes:
            p = s.GetCenter()
            if s.GetShape() is SHAPE_T_CIRCLE:
                w = h = s.GetRadius() * 2
            else:
                w, h = map(abs, s.GetEnd() - s.GetStart())
            s.SetPolyPoints(wxPoint_Vector([p + i for i in octastar(w, h)[1::2]]))
            s.SetShape(SHAPE_T_POLY)

    # handle pads & vias
    for v, tracks in connections.items():

        # get directly connected track segments by layer
        for t in connectivity.GetConnectedTracks(v):
            tracks[t.GetLayer()].append(t)

        if type(v) is PAD:
            attr = v.GetAttribute()

            # skip SMD/NPTH/CONN pads if not explicitly selected
            if notselected and attr is not PAD_ATTRIB_PTH:
                continue

            size = v.GetSize()
            drill = v.GetDrillSize()
            keepout = attr is PAD_ATTRIB_NPTH and drill == size

            if keepout:
                try:
                    c = v.GetLocalClearanceOverrides(None) * 2
                except AttributeError:
                    c = v.GetClearance() * 2
                star = octastar(size.x + c, size.y + c)[1::2]  # octagon, no fillet
            else:
                star = octastar(*size)

            offset = v.GetOffset()
            if any(offset):
                star = (offset + i for i in star)

            try:
                rotation = v.GetOrientation().AsRadians()
            except AttributeError:
                rotation = v.GetOrientationRadians()
            if rotation:
                c, s = cos(rotation), sin(rotation)
                star = (wxPoint(round(c*i[0] + s*i[1]), round(c*i[1] - s*i[0])) for i in star)

            if keepout:
                position = v.GetPosition()
                points = [position + i for i in star]
                z = ZONE(keepoutdefaults)
                z.SetLayerSet(v.GetLayerSet())
                z.AddPolygon(wxPoint_Vector(points))
                z.SetLocked(v.IsLocked())

                board.Add(z)
                continue

        else:  # is VIA
            # potential coordinates suitable for zones
            star = octastar(v.GetWidth(), v.GetWidth())

        # generate fillet zone for each layer
        position = v.GetPosition()
        star = [position + i for i in star]
        hitpos = star[::2]  # outer points of star only
        net = v.GetNetCode()

        for layer, segments in tracks.items():

            # figure out where track segments connect
            fillet = [any(t.HitTest(p) for t in segments) for p in hitpos]

            # pads change their shape, vias don't
            if not any(fillet) and type(v) is PAD:
                continue

            # select required coordinates
            points = [
                star[i*2 + j]
                for i in range(8)
                    for j in range(2)
                        if fillet[i] != (j and not fillet[i-7])
            ]

            z = ZONE(zonedefaults)
            z.SetNetCode(net)
            z.SetLayer(layer)
            z.AddPolygon(wxPoint_Vector(points))
            z.SetLocked(v.IsLocked())

            # copy highest priority if inside another zone (best guess)
            try:
                p = [h.GetAssignedPriority() for h in zones[net] if h.HitTestFilledArea(layer, position)]
                if p:
                    z.SetAssignedPriority(max(p))
            except AttributeError:
                try:
                    p = [h.GetPriority() for h in zones[net] if h.HitTestFilledArea(layer, position)]
                except TypeError:
                    p = [h.GetPriority() for h in zones[net] if h.IsOnLayer(layer) and h.HitTestFilledArea(position)]
                if p:
                    z.SetPriority(max(p))

            board.Add(z)

        # change pad shape
        if type(v) is PAD:
            shape = v.GetShape()
            # rects support thermal but custom shape has nicer solder mask
            #if shape is PAD_SHAPE_RECT:
            #    if STROKE_WIDTH != 0:
            #        v.SetShape(PAD_SHAPE_ROUNDRECT)
            #        v.SetRoundRectCornerRadius(STROKE_WIDTH * 0.5)
            #elif STROKE_WIDTH == 0 and (attr is PAD_ATTRIB_PTH or shape is not PAD_SHAPE_ROUNDRECT
            #                                                   or v.GetRoundRectRadiusRatio() >= 0.20):
            #    v.SetShape(PAD_SHAPE_CHAMFERED_RECT)
            #    v.SetChamferRectRatio(chamferRatio(*size))
            #    v.SetChamferPositions(RECT_CHAMFER_ALL)
            #el
            if STROKE_WIDTH == 0 and shape is PAD_SHAPE_RECT and not any(offset):
                v.SetShape(PAD_SHAPE_CUSTOM)
                v.SetCustomShapeInZoneOpt(CUST_PAD_SHAPE_IN_ZONE_CONVEXHULL)  # for KiCad 5.1
                v.SetAnchorPadShape(PAD_SHAPE_RECT)  # no primitive necessary
                v.DeletePrimitivesList()
            elif attr is PAD_ATTRIB_PTH or shape is not PAD_SHAPE_ROUNDRECT or v.GetRoundRectRadiusRatio() >= 0.20:
                v.SetShape(PAD_SHAPE_CUSTOM)
                v.SetCustomShapeInZoneOpt(CUST_PAD_SHAPE_IN_ZONE_CONVEXHULL)  # for KiCad 5.1
                v.SetAnchorPadShape(PAD_SHAPE_CIRCLE)  # tentatively
                stroke = STROKE_WIDTH if min(size) > STROKE_WIDTH else 0  # no rounded corners if too small
                star = octastar(size.x, size.y, stroke)  # inset
                userect = size.x != size.y or drill.x != drill.y  # v.SetSize() below also changes size.x and size.y
                if any(offset) or rotation or attr is not PAD_ATTRIB_PTH:
                    star = [offset + i for i in star]
                    v.SetOffset(wxPoint(0, 0))
                    v.SetSize(wxSize(*2*[min(size)]))
                if userect:
                    v.SetAnchorPadShape(PAD_SHAPE_RECT)
                    v.SetSize(wxSize(drill.x + 2, drill.y + 2))  # just not equal to hole size
                points = star[2::4] if shape is PAD_SHAPE_RECT else star[1::2]  # rect/octa
                v.DeletePrimitivesList()
                try:
                    v.AddPrimitivePoly(wxPoint_Vector(points), stroke, True)
                except AttributeError:
                    v.AddPrimitive(wxPoint_Vector(points), stroke)

    # same as Fill All Zones "B" (unfortunately; should do something like 8d05ca59)
    try:
        ZONE_FILLER(board).Fill(board.Zones())
    except NameError:
        pass  # ignore if not imported


def registerPlugin():
    """Register as Action Plugin."""
    class OktizerActionPlugin(ActionPlugin):
        def defaults(self):
            self.name = "Oktizer " + __version__
            self.category = "PCB/Footprint Modification"
            self.description = "Change (selected or all round through-hole) pads and vias to octagons."
            self.show_toolbar_button = True

        def Run(self):
            oktize()

    OktizerActionPlugin().register()


# After adding footprints:
#     Export → Footprints to New Library…
#     File → Archive Footprints → Create New Library and Archive Footprints…

def addProtoPadFootprints(board=None):
    """Add footprints with prototype pads to board."""
    if chamferRatio() != 1 - .5**.5:
        raise NotImplementedError("irregular octagon")

    if board is None:
        board = GetBoard()

    sizes = [
        ((1 + 2**.5) * side, (1 + 2**.5 * (1 - truncate*.5)) * side, drill, side, offcenter)
        for side, drill, truncate, offcenter in (
            (0.625, 0.85,  1, 0),
            (0.50,  0.70,  0, 0),
            (0.60,  0.90,  0, 0),
            (0.625, 0.85,  0, 0),
            (0.65,  0.975, 0, 0),
            (1.00,  1.00,  1, 0),
            (1.00,  1.00,  1, 1),
            (0.75,  1.00,  0, 0),
            (0.75,  1.30,  0, 0),
            (1.00,  1.00,  0, 0),
            (1.25,  1.20,  0, 0),
        )
    ]

    SHAPES = PAD_SHAPE_OVAL, PAD_SHAPE_RECT, PAD_SHAPE_CUSTOM, PAD_SHAPE_CHAMFERED_RECT,
    SHAPETEXT = (
        ("obround", "rectangular", "octagonal", "chamfered"),
        ("round", "square", "octagonal", "chamfered"),
        ("", "Rect_", "Octa_", "Chamfer_"),
    )

    for l, shape in enumerate(SHAPES):
        if shape is NotImplemented:
            continue
        for j, k in enumerate(sizes):
            m, offcenter = k[:4], k[4]
            for customary in (0, 1):
                name = "Pad_" + SHAPETEXT[2][l]
                desc = "Single " + SHAPETEXT[m[0] == m[1]][l] + " THT pad "
                if customary:
                    metric = tuple(i * 254 / 250 for i in m)
                    custom = tuple(i / 25.0 for i in m)
                    if shape is PAD_SHAPE_OVAL and m[0] == m[1]:
                        name += "%.3fin_Drill%.3fin" % custom[1:3]
                        desc += "⌀ %.3f in, hole ⌀ %.3f in, nominal trace width %.3f in" % custom[1:]
                    else:
                        name += "%.3fx%.3fin_Drill%.3fin" % custom[:3]
                        desc += "%.3f×%.3f in², hole ⌀ %.3f in, nominal trace width %.3f in" % custom
                else:
                    metric = m
                    custom = tuple(i / 25.4 for i in m)
                    if shape is PAD_SHAPE_OVAL and m[0] == m[1]:
                        name += "%.2fmm_Drill%.1fmm" % metric[1:3]
                        desc += "⌀ %.2f mm, hole ⌀ %g mm, nominal trace width %g mm" % metric[1:]
                    else:
                        name += "%.2fx%.2fmm_Drill%.1fmm" % metric[:3]
                        desc += "%.2f×%.2f mm², hole ⌀ %g mm, nominal trace width %g mm" % metric
                if offcenter:
                    name += "_Offset"
                    desc += ", offset hole"

                o, w, h, d, s = mm2iu((metric[0]-metric[1]) * 0.5, *metric)
                n = "1"
                f = FOOTPRINT(board)
                f.SetFPID(LIB_ID("", name))
                f.SetValue(name)
                f.Value().SetLayer(F_Fab)
                f.Value().SetTextSize(wxSize(*mm2iu(1, 1)))
                f.Value().SetPos0(wxPoint(*mm2iu(0, 3)))
                f.SetReference("REF**")
                f.Reference().SetTextSize(wxSize(*mm2iu(1, 1)))
                f.Reference().SetPos0(wxPoint(*mm2iu(0, -3)))
                f.SetAttributes(FP_VIRTUAL)
                pos = wxPoint(*mm2iu(j * 25 + 25, (l * 2 + customary) * 10 + 25))
                f.SetPosition(pos)
                f.SetDescription(desc)

                # main pad
                p = PAD(f)
                p.SetSize(wxSize(w, h))
                p.SetDrillSize(wxSize(d, d))
                offset = wxPoint(o if offcenter else 0, 0)
                p.SetOffset(offset)
                p.SetAttribute(PAD_ATTRIB_PTH)
                p.SetShape(PAD_SHAPE_CIRCLE if shape is PAD_SHAPE_OVAL and not o else shape)
                if shape is PAD_SHAPE_CUSTOM:
                    p.SetAnchorPadShape(PAD_SHAPE_CIRCLE)
                    p.SetSize(wxSize(h, h))
                    p.SetCustomShapeInZoneOpt(CUST_PAD_SHAPE_IN_ZONE_CONVEXHULL)
                    p.SetAnchorPadShape(PAD_SHAPE_CIRCLE)
                    points = octastar(w, h)[1::2]  # octagon
                    if any(offset):
                        points = [offset + i for i in points]
                        p.SetOffset(wxPoint(0, 0))
                    try:
                        p.AddPrimitivePoly(wxPoint_Vector(points), 0, True)
                    except AttributeError:
                        p.AddPrimitive(wxPoint_Vector(points), 0)
                elif shape is PAD_SHAPE_CHAMFERED_RECT:
                    p.SetChamferRectRatio(chamferRatio(w, h))
                    p.SetChamferPositions(RECT_CHAMFER_ALL)
                p.SetRoundRectCornerRadius(0)
                #try:
                #    p.SetLayerSet(PAD.PTHMask())
                #except AttributeError:
                #    p.SetLayerSet(PAD.StandardMask())
                p.SetPos0(wxPoint(0, 0))
                p.SetName(n)
                f.Add(p)

                if o:
                    # 4 aux pads as magnets for 45°/90° connections
                    for layers in map(LSET, (F_Cu, B_Cu)):
                        for off in (w-h if offcenter else -o, o):
                            q = PAD(f)
                            q.SetSize(wxSize(d, d))
                            q.SetDrillSize(wxSize(0, 0))
                            q.SetOffset(wxPoint(0, 0))
                            q.SetShape(PAD_SHAPE_CIRCLE)
                            q.SetAttribute(PAD_ATTRIB_CONN)
                            q.SetLayerSet(layers)
                            q.SetPos0(wxPoint(off, 0))
                            q.SetName(n)
                            f.Add(q)

                if shape is PAD_SHAPE_CHAMFERED_RECT and 1.6 < metric[1] < 2.0 and metric[2] < 1.1:
                    # silkscreen
                    g = FP_SHAPE(f, SHAPE_T_POLY)
                    g.SetFilled(False)
                    g.SetLayer(F_SilkS)
                    g.SetWidth(mm2iu(0.12))
                    pitch = mil2iu(100) if customary else mm2iu(2.50)
                    g.SetPolyPoints(wxPoint_Vector([offset + i for i in octastar(w, h, min(w, h) - pitch)[1::2]]))
                    f.Add(g)

                try:
                    f.SetLastEditTime()
                    f.SetTimeStamp(f.GetLastEditTime())
                except (TypeError, AttributeError):
                    try:
                        import time
                        f.SetLastEditTime(int(time.time()))
                        # using UUID
                    except (TypeError, AttributeError):
                        pass  # using UUID
                board.Add(f)

    return board


if __name__ == "__main__":
    from sys import argv
    args = argv[1:] or ["Elektuur"]
    filename = args[0] + "." + IO_MGR.GetFileExtension(IO_MGR.KICAD_SEXP)
    if addProtoPadFootprints(BOARD()).Save(filename):
        print('Created "%s".' % filename)
else:
    registerPlugin()
